const { Sequelize, DataTypes } = require("sequelize");
const db = require("../config/db");
const bcrypt = require("bcryptjs");

// User Model
const User = db.define(
  "User",
  {
    id: {
      type: DataTypes.INTEGER,
      primaryKey: true,
      autoIncrement: true,
    },
    first_name: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    last_name: {
      type: DataTypes.STRING,
      allowNull: true,
    },
    email: {
      type: DataTypes.STRING,
      allowNull: false,
      unique: true,
      validate: { isEmail: true },
    },
    password: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    role: {
      type: DataTypes.ENUM("user", "admin", "superadmin"),
      defaultValue: "user",
    },
    approved: {
      type: DataTypes.BOOLEAN,
      defaultValue: false,
    },
  },
  {
    timestamps: true, // includes createdAt & updatedAt
  }
);

// Hash password before creating a user
User.beforeCreate(async (user) => {
  user.password = await bcrypt.hash(user.password, 10);
});

// Optional: Hash password on update too
User.beforeUpdate(async (user) => {
  if (user.changed("password")) {
    user.password = await bcrypt.hash(user.password, 10);
  }
});

// Member Model
const Member = db.define(
  "Member",
  {
    id: { type: DataTypes.INTEGER, primaryKey: true, autoIncrement: true },
    first_name: { type: DataTypes.STRING, allowNull: false },
    last_name: { type: DataTypes.STRING, allowNull: false },
    email: { type: DataTypes.STRING, allowNull: false, unique: true },
    phone_number: { type: DataTypes.STRING, allowNull: false },
    address: DataTypes.STRING,
    date_of_birth: DataTypes.DATE,
    membership_status: { type: DataTypes.STRING, allowNull: false },
  },
  { timestamps: true }
);

// Event Model
const Event = db.define(
  'Event',
  {
    eventId: { // Changed from 'id' to 'eventId' for consistency with frontend
      type: DataTypes.INTEGER,
      primaryKey: true,
      autoIncrement: true
    },
    title: { 
      type: DataTypes.STRING, 
      allowNull: false 
    },
    description: { 
      type: DataTypes.TEXT 
    },
    date: {  // Changed from 'event_date' to 'date'
      type: DataTypes.DATE, 
      allowNull: false 
    },
    venue: {  // Changed from 'location' to 'venue'
      type: DataTypes.STRING
    },
    image: {  // Changed from 'banner_url' to 'bannerUrl'
      type: DataTypes.STRING
    },
    startTime: {  // Added to match frontend
      type: DataTypes.TIME
    },
    endTime: {  // Added to match frontend
      type: DataTypes.TIME
    }
  },
  { 
    timestamps: false, // Set to false since the frontend doesn't require timestamps
    tableName: 'events'
  }
);

// Visitor Model
const Visitor = db.define(
  'Visitor', {
    visitor_id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true, // This will be the unique primary key for the table
  },
  id_number: {
      type: DataTypes.INTEGER,
      allowNull: false,
      unique: true, // This ensures each id_number is unique but not a primary key
  },  
      name: {
          type: DataTypes.STRING,
          allowNull: false
      },
      gender: {
          type: DataTypes.STRING,
          allowNull: false
      },
      email: {
          type: DataTypes.STRING,
          allowNull: false,
          unique: true
      },
      mobile: {
          type: DataTypes.STRING,
          allowNull: false
      },
      date_of_visit: {
          type: DataTypes.DATE,
          allowNull: false
      }
  }, {
      timestamps: false,
      tableName: 'visitors'
  });


// Prayer Request Model

const Prayer = sequelize.define(
  "Prayer",
  {
    requestId: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    memberName: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    description: {
      type: DataTypes.STRING,
      allowNull: false,
    },
    requestStatus: {
      type: DataTypes.STRING,
      allowNull: false,
      defaultValue: "Pending",
    },
  },
  {
    timestamps: true,
    tableName: "prayers",
  }
);

// Export Models
module.exports = { User, Member, Event, Visitor, Prayer };
