const { Sequelize, DataTypes } = require("sequelize");
const db = require("../config/db");
const bcrypt = require("bcryptjs");

// User Model
const User = db.define(
  "User",
  {
    id: { type: DataTypes.INTEGER, primaryKey: true, autoIncrement: true },
    first_name: {
      type: DataTypes.STRING,
      allowNull: true, // Optional first name for signup
    },
    last_name: {
      type: DataTypes.STRING,
      allowNull: true, // Optional last name for signup
    },
    email: { 
      type: DataTypes.STRING, 
      allowNull: false, 
      unique: true, // Ensure email is unique
    },
    password: { 
      type: DataTypes.STRING, 
      allowNull: false,
    },
  },
  { timestamps: true } // Adds createdAt & updatedAt
);

// Hash the password before saving it to the database
User.beforeCreate(async (user) => {
  user.password = await bcrypt.hash(user.password, 10);
});

// Member Model
const Member = db.define(
  "Member",
  {
    id: { type: DataTypes.INTEGER, primaryKey: true, autoIncrement: true },
    first_name: { type: DataTypes.STRING, allowNull: false },
    last_name: { type: DataTypes.STRING, allowNull: false },
    email: { type: DataTypes.STRING, allowNull: false, unique: true },
    phone_number: { type: DataTypes.STRING, allowNull: false },
    address: DataTypes.STRING,
    date_of_birth: DataTypes.DATE,
    membership_status: { type: DataTypes.STRING, allowNull: false },
  },
  { timestamps: true }
);

// Event Model
const Event = db.define(
  'Event',
  {
    eventId: { // Changed from 'id' to 'eventId' for consistency with frontend
      type: DataTypes.INTEGER,
      primaryKey: true,
      autoIncrement: true
    },
    title: { 
      type: DataTypes.STRING, 
      allowNull: false 
    },
    description: { 
      type: DataTypes.TEXT 
    },
    date: {  // Changed from 'event_date' to 'date'
      type: DataTypes.DATE, 
      allowNull: false 
    },
    venue: {  // Changed from 'location' to 'venue'
      type: DataTypes.STRING
    },
    image: {  // Changed from 'banner_url' to 'bannerUrl'
      type: DataTypes.STRING
    },
    startTime: {  // Added to match frontend
      type: DataTypes.TIME
    },
    endTime: {  // Added to match frontend
      type: DataTypes.TIME
    }
  },
  { 
    timestamps: false, // Set to false since the frontend doesn't require timestamps
    tableName: 'events'
  }
);

// Visitor Model
const Visitor = db.define(
  'Visitor', {
    visitor_id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true, // This will be the unique primary key for the table
  },
  id_number: {
      type: DataTypes.INTEGER,
      allowNull: false,
      unique: true, // This ensures each id_number is unique but not a primary key
  },  
      name: {
          type: DataTypes.STRING,
          allowNull: false
      },
      gender: {
          type: DataTypes.STRING,
          allowNull: false
      },
      email: {
          type: DataTypes.STRING,
          allowNull: false,
          unique: true
      },
      mobile: {
          type: DataTypes.STRING,
          allowNull: false
      },
      date_of_visit: {
          type: DataTypes.DATE,
          allowNull: false
      }
  }, {
      timestamps: false,
      tableName: 'visitors'
  });


// Prayer Request Model
const PrayerRequest = db.define(
  "PrayerRequest",
  {
    id: { type: DataTypes.INTEGER, primaryKey: true, autoIncrement: true },
    member_id: {
      type: DataTypes.INTEGER,
      allowNull: false,
      references: { model: Member, key: "id" }, // Foreign key reference
      onDelete: "CASCADE",
    },
    request_text: { type: DataTypes.TEXT, allowNull: false },
    request_status: { type: DataTypes.STRING, defaultValue: "Pending" },
    is_anonymous: { type: DataTypes.BOOLEAN, defaultValue: false },
  },
  { timestamps: true }
);

// Export Models
module.exports = { User, Member, Event, Visitor, PrayerRequest };
